﻿/*
	Original Base Station implementation thanks to Joseph Scaduto
	Modified for server update via SCP
	
	Andrew Schulze
*/

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.IO;
using System.IO.Ports;
using System.Linq;
using System.Text;
using System.Threading;
using System.Windows.Forms;


public struct BUFFER
{
    public static bool flag;
    //if flag is true the value has not be read, if flag is false the value can be replaced
    public bool fReadWrite
    {
        get
        {
            return flag;
        }
        set
        {
            flag = value;
        }
    }

    public static string buf;
    public string Buffer
    {
        get
        {
            return buf;
        }
        set
        {
            buf = value;
        }
    }
}

namespace BaseStation
{
    public partial class frmBSDisplay : Form
    {
        public static string COMPort;
        public static bool fThread1;
        public static BUFFER[] circleBuffer = new BUFFER[10];

        public static double ambient_temp;      /* ambient temperature      */
        public static double cum_hyd_consump;   /* cumulative h2 consumed   */
        public static double hyd_concen;        /* hydrogen concentration   */
        public static double hyd_press;         /* hydrogen pressure        */
        public static int fail_code;            /* failure code             */
        public static double oxy_concen;        /* oxygen concentration     */
        public static double purge_cell_volt;   /* purge cell voltage       */
        public static double stack_curr;        /* cell stack current       */
        public static double stack_temp;        /* cell stack temperature   */
        public static double stack_volt;        /* cell stack voltage       */
        public static int status_code;          /* status code              */
        public static DateTime lastUpdate;        /* timestamp                */
        public static byte warn_codes;          /* warning codes (bitmap)   */
        public static double fuel_eff;          /* timestamp                */
        public static double speed;             /* timestamp                */

        public static DateTime Time_start;

        /* fuel cell status codes   */
        string[] status_codes = 
            {
            "Standby",                          /* standby                  */
            "Start Up",                         /* start up                 */
            "Normal Operation",                 /* normal operation         */
            "Warning",                          /* warning                  */
            "Normal Shut Down",                 /* normal shut down         */
            "Failure Shut Down",                /* failure shut down        */
            "Non Restartable"                   /* non-restartable          */
            };

        /* fuel cell fail codes     */
        string[] fail_codes =   
            {
            "Normal Operation",                 /* normal operation         */
            "High Fuel Cell Stack Temperature", /* high stack temperature   */
            "Low Fuel Cell Stack Voltage",      /* low stack voltage        */
            "High Fuel Cell Stack Current",     /* high stack current       */
            "Low Cell Voltage",                 /* low cell voltage         */
            "Low Fuel Pressure",                /* low fuel pressure        */
            "Fuel Leak Detected",               /* fuel leak detected       */
            "Low Oxygen Concentration",         /* low oxygen concentration */
            "Low Ambient Temperature",          /* low ambient temperature  */
            "Low Purge Cell Voltage",           /* low purge cell voltage   */
            "Low Battery Voltage",              /* low battery voltage      */
            "Startup Time Expired",             /* startup time expired     */
            "Self Test Fault",                  /* self test fault          */
            "General Software Fault",           /* general software fault   */
            "Spurious Interrupt Fault"          /* spurious interrupt fault */
            };

        /* fuel cell warning codes  */
        string[] warning_codes =
            {
            "No Warnings",                      /* no warnings              */
            "High Fuel Cell Stack Temperature", /* high stack temp          */
            "Low Fuel Cell Stack Voltage",      /* low stack voltage        */
            "High Fuel Cell Stack Current",     /* high stack current       */
            "Low Fuel Pressure",                /* low fuel pressure        */
            "Fuel Leak",                        /* fuel leak                */
            "Low Oxygen Concentration",         /* low oxygen concentration */
            "Low Purge Cell Voltage"            /* low purge cell voltage   */
            };

        public frmBSDisplay()
        {
            InitializeComponent();

            string[] ports = SerialPort.GetPortNames();
            Array.Sort(ports);
            Array.Reverse(ports);
            cmbxCOMPort.Items.AddRange(ports);
            cmbxCOMPort.SelectedIndex = 0;

            cum_hyd_consump = 0;

            ThreadStart task1 = new ThreadStart(SerialMonitor);
            Thread thread1 = new Thread(task1);
            thread1.IsBackground = true;

            ThreadStart task2 = new ThreadStart(DisplayUpdater);
            Thread thread2 = new Thread(task2);
            thread2.IsBackground = true;

            thread1.Start();
            thread2.Start();
        }

        protected override void OnPaint(PaintEventArgs pe)
        {
            Graphics g = pe.Graphics;
            Rectangle rect = new Rectangle();
            rect = Screen.GetBounds(this);
            LinearGradientBrush lBrush = new LinearGradientBrush(rect, Color.Black, Color.Gray, LinearGradientMode.ForwardDiagonal);
            g.FillRectangle(lBrush, rect);
        }

        public void SerialMonitor()
        {
            SerialPort sp = new SerialPort();

            while (true)
            {
                sp.PortName = COMPort;
                sp.BaudRate = 9600;
                try
                {
                    sp.Open();
                }
                catch (Exception ex)
                {
                    txtStatus.Invoke(new UpdateTxtStatusCallback(this.UpdateTxtStatus), new object[] { "ERROR: " + ex.Message + " Check that correct COM port is selected and that no terminals are monitoring that COM port."  });
                }

                fThread1 = true;
                if (sp.IsOpen)
                //if(true)
                {
                    while (fThread1)
                    {
                        for (int i = 0; i < 10; i++)
                        {
                            //read till NEW LINE char
                            circleBuffer[i].Buffer = sp.ReadLine();
                            circleBuffer[i].fReadWrite = true;
                        }
                    }
                }
                sp.Close();
            }
        }

        public delegate void UpdateGaugeEfficencyCallback(float eff);
        private void UpdateGaugeEfficency(float eff)
        {
            if (eff < gaugeEfficiency.MinValue)
                eff = gaugeEfficiency.MinValue;
            if (eff > gaugeEfficiency.MaxValue)
                eff = gaugeEfficiency.MaxValue;
            gaugeEfficiency.Value = eff;
        }

        public delegate void UpdateGaugeSpeedCallback(float spd);
        private void UpdateGaugeSpeed(float spd)
        {
            if (spd < gaugeSpeed.MinValue)
                spd = gaugeSpeed.MinValue;
            if (spd > gaugeSpeed.MaxValue)
                spd = gaugeSpeed.MaxValue;
            gaugeSpeed.Value = spd;
        }

        public delegate void UpdateGaugeTempCallback(float tmp);
        private void UpdateGaugeTemp(float tmp)
        {
            if (tmp < gaugeTemp.MinValue)
                tmp = gaugeTemp.MinValue;
            if (tmp > gaugeTemp.MaxValue)
                tmp = gaugeTemp.MaxValue;
            gaugeTemp.Value = tmp; 
        }

        public delegate void UpdateGaugeFuelCallback(float fuel);
        private void UpdateGaugeFuel(float fuel)
        {
            if (fuel < gaugeFuel.MinValue)
                fuel = gaugeFuel.MinValue;
            if (fuel > gaugeFuel.MaxValue)
                fuel = gaugeFuel.MaxValue;
            gaugeFuel.Value = fuel;
        }

        public delegate void UpdateTxtStatusCallback(string display);
        private void UpdateTxtStatus(string display)
        {
            txtStatus.Text = display;
        }

        public void DisplayUpdater()
        {
            while(true)
            {
                for (int i = 0; i < 10; i++)
                {
                    if (circleBuffer[i].fReadWrite == true)
                    //if (true)    //uncomment for testing
                    {
                        //if no parseing errors
                        if (parseMetrics(circleBuffer[i].Buffer) == 0)
                        {
                            circleBuffer[i].fReadWrite = false;
                            updateDisplay();
                        }
                        else
                        {
                            circleBuffer[i].fReadWrite = false;
                        }
                    }
                }
            }
        }

		/*
		*************************************************
		THIS IS START OF CODE DEVELOPED BY ANDREW SCHULZE
		*************************************************
		*/
		
        static void updateServer(String metrics)
        {
            //Directory.GetCurrentDirectory() -- use this in BaseStation to get current directory assuming pscp, key.ppk are stored in the same location as BaseStation

            string exe = Directory.GetCurrentDirectory() + "\\pscp.exe";
            string ppk = Directory.GetCurrentDirectory() + "\\key.ppk";
            string srcfile = Directory.GetCurrentDirectory() + "\\index.html";
            string dest = "PAWPRINT@bengal.missouri.edu:www/"; //change PAWPRINT to your pawprint to access files in your account on this server
            int regkey = 0;
            int i = 0;

            string[] parts = metrics.Split(',');

            Microsoft.Win32.RegistryKey subKey = Microsoft.Win32.Registry.CurrentUser.OpenSubKey("Software\\SimonTatham\\PuTTY\\SshHostKeys");
            if (subKey != null)
            {
               if (subKey.GetValue("rsa2@22:bengal.missouri.edu") != null)
                {
                    //Console.Write("Registry Value exists");
                    regkey = 1;
                    subKey.Close();
                }
            }


            // create a writer and open the file
            TextWriter tw = new StreamWriter(srcfile);

            // write a line of text to the file
            DateTime baseTime = new DateTime(1970, 1, 1, 0, 0, 0);
            DateTime nowInUTC = DateTime.UtcNow;
            tw.WriteLine(((nowInUTC - baseTime).Ticks / 10000) + "," + parts[13] + "," + parts[14] + "," + (Convert.ToChar(parts[5]) - 48));
            tw.WriteLine();
            tw.Close();

			LaunchCommandLineApp(exe, ppk, srcfile, dest, ref regkey);
        }

        static void LaunchCommandLineApp(string exe, string ppk, string srcfile, string dest, ref int regkey)
        {
            // Use ProcessStartInfo class
            ProcessStartInfo startInfo = new ProcessStartInfo();

            // Set ProcessStartInfo object parameters
            startInfo.CreateNoWindow = true;
            startInfo.UseShellExecute = false;
            startInfo.FileName = exe;
            startInfo.WindowStyle = ProcessWindowStyle.Hidden;
            startInfo.RedirectStandardError = false;
            startInfo.RedirectStandardOutput = true;
            startInfo.RedirectStandardInput = true;
            startInfo.Arguments = " -i " + ppk + " " + srcfile + " " + dest;

            try
            {
                // Start the process with the info we specified.
                // Call WaitForExit and then the using statement will close.
                using (Process exeProcess = Process.Start(startInfo))
                {
                    // Check for existence of registry key.
                    // This will take care of the y/n prompt if key does not exist.
                    if (regkey == 0)
                    {
                        // Send a yes to console so that the ssh fingerprint is stored.
                        exeProcess.StandardInput.Write('y');
                        exeProcess.WaitForExit();

                        // Recreate RegistryKey for this function.
                        Microsoft.Win32.RegistryKey subKey = Microsoft.Win32.Registry.CurrentUser.OpenSubKey("Software\\SimonTatham\\PuTTY\\SshHostKeys");
                        // Verifies that key was created.
                        if (subKey.GetValue("rsa2@22:bengal.missouri.edu") != null)
                        {
                            //Console.Write("Registry Value exists");
                            regkey = 1;
                            subKey.Close();
                        }
                    }
                    // Registry key exists so carry on normally.
                    else
                    {
                        exeProcess.WaitForExit();
                    }
                }
            }
            catch
            {
                Console.WriteLine("Error\r\n");
            }
        }
		
		/*
		***********************************************
		THIS IS END OF CODE DEVELOPED BY ANDREW SCHULZE
		***********************************************
		*/

        public int parseMetrics(string metrics)
        {
            //metrics = "AA,25.31,.03,4.94,5.22,0,20.97,1.97,0.16,28.16,45.67,4,3,30.00,0.08";
            string[] parts = metrics.Split(',');

            if ((parts[0] == "AA") && (parts.Length == 15))
            {
                try
                {
                    ambient_temp = Double.Parse(parts[1]);

                    cum_hyd_consump = Math.Round(Double.Parse(parts[2]), 5);

                    hyd_concen = Double.Parse(parts[3]);

                    hyd_press = Double.Parse(parts[4]);

                    fail_code = Convert.ToInt32(Convert.ToChar(parts[5]) - 48);

                    oxy_concen = Double.Parse(parts[6]);

                    purge_cell_volt = Double.Parse(parts[7]);

                    stack_curr = Double.Parse(parts[8]);

                    stack_temp = Double.Parse(parts[9]);

                    stack_volt = Double.Parse(parts[10]);

                    status_code = Convert.ToInt32(Convert.ToChar(parts[11]) - 48);

                    warn_codes = Convert.ToByte(Convert.ToChar(parts[12]) - 48);

                    speed = Double.Parse(parts[13]);

                    fuel_eff = Double.Parse(parts[14]);
                }
                catch (Exception e)
                {
                    return 1;
                }

                updateServer(metrics); //new function (not in original implementation)
            }
            return 0;
        }

        public void updateDisplay()
        {
            lastUpdate = DateTime.Now;

            //update dials
            gaugeTemp.Invoke(new UpdateGaugeTempCallback(this.UpdateGaugeTemp), new object[] { ((float)stack_temp) });
            gaugeEfficiency.Invoke(new UpdateGaugeEfficencyCallback(this.UpdateGaugeEfficency), new object[] { ((float)fuel_eff) });
            gaugeSpeed.Invoke(new UpdateGaugeSpeedCallback(this.UpdateGaugeSpeed), new object[] { ((float)speed) });
            gaugeFuel.Invoke(new UpdateGaugeFuelCallback(this.UpdateGaugeFuel), new object[] { ((float)cum_hyd_consump) });

            //update txtStatus box
            string display;
            display = "STATUS: " + status_codes[status_code] + "\r\n";
            display += "FAIL: " + fail_codes[fail_code] + "\r\n";
            display += "WARNINGS: " + decode_Warnings() + "\r\n";
            display += "Ambient Temp: " + Convert.ToString(ambient_temp) + " °C\r\n";
            display += "Cumulative H2 Consump: " + String.Format("{0:0.###}",cum_hyd_consump) + " L\r\n";
            display += "H2 Concentration: " + Convert.ToString(hyd_concen) + " ppm\r\n";
            display += "H2 Pressure: " + Convert.ToString(hyd_press) + " bar\r\n";
            display += "O2 Concentration: " + Convert.ToString(oxy_concen) + " %\r\n";
            display += "Purge Cell Volt: " + Convert.ToString(purge_cell_volt) + " V\r\n";
            display += "Stack Current: " + Convert.ToString(stack_curr) + " A\r\n";
            display += "Stack Temp: " + Convert.ToString(stack_temp) + " °C\r\n";
            display += "Stack Voltage: " + Convert.ToString(stack_volt) + " V\r\n";
            display += "Last Updated: " + Convert.ToString(lastUpdate) + "\r\n";

            txtStatus.Invoke(new UpdateTxtStatusCallback(this.UpdateTxtStatus), new object[] { display });
        }

        public string decode_Warnings()
        {
            string strWarnings = "";
            byte code = Convert.ToByte(warn_codes);
            if (code == 0)
            {
                return warning_codes[0];
            }
            for (int i = 1; i < warning_codes.Length; i++)
            {
                byte mask = Convert.ToByte(Math.Pow(2, i - 1));
                if ((code & mask) == mask)
                {
                    if (strWarnings.Length == 0)
                    {
                        strWarnings = warning_codes[i];
                    }
                    else
                    {
                        strWarnings += ", " + warning_codes[i];
                    }
                }
            }
            return strWarnings;
        }

        private void frmBSDisplay_Resize(object sender, EventArgs e)
        {
            formatDisplay();
        }

        private void frmBSDisplay_Shown(object sender, EventArgs e)
        {
            formatDisplay();
        }

        private void formatDisplay()
        {
            groupBox1.Width = this.Width - 30;
            groupBox1.Height = this.Height - 30;

            if ((groupBox1.Height / groupBox1.Width) < (2.0 / 3.0))
                groupBox1.Height = Convert.ToInt16(groupBox1.Width * 2.0 / 3.0);
            if ((groupBox1.Height / groupBox1.Width) > (2.0 / 3.0))
                groupBox1.Width = Convert.ToInt16(groupBox1.Height * 3.0 / 2.0);

            groupBox1.Location = new Point((this.Width - groupBox1.Width) / 4, (this.Height - groupBox1.Height) / 20);

            int padding = 10;
      
            lblCOMPort.Location = new Point((groupBox1.Width / 16 + groupBox1.Location.X), (groupBox1.Height / 2 + lblCOMPort.Height + groupBox1.Location.Y));

            cmbxCOMPort.Location = new Point((groupBox1.Location.X + 20), (groupBox1.Height / 2 + cmbxCOMPort.Height + lblCOMPort.Height + 5 + groupBox1.Location.Y));

            this.txtStatus.Size = new Size((groupBox1.Width * 2 / 3), (groupBox1.Height * 2 / 5));
            this.txtStatus.Location = new Point((groupBox1.Width / 6 + groupBox1.Location.X), (groupBox1.Height - txtStatus.Height) - 50 + groupBox1.Location.Y);

            this.gaugeTemp.Size = new Size((groupBox1.Height / 4) - padding, (groupBox1.Height / 4) - padding);
            this.gaugeTemp.Location = new Point(groupBox1.Location.X, (groupBox1.Height / 4) + padding / 2 + groupBox1.Location.Y);

            this.gaugeEfficiency.Size = new Size((groupBox1.Height / 2) - padding, (groupBox1.Height / 2) - padding);
            this.gaugeEfficiency.Location = new Point((groupBox1.Width / 6 + groupBox1.Location.X) + padding / 2, padding / 2 + groupBox1.Location.Y);

            this.gaugeSpeed.Size = new Size((groupBox1.Height / 2) - padding, (groupBox1.Height / 2) - padding);
            this.gaugeSpeed.Location = new Point((groupBox1.Width / 2 + groupBox1.Location.X) + padding / 2, padding / 2 + groupBox1.Location.Y);

            this.gaugeFuel.Size = new Size((groupBox1.Height / 4) - padding, (groupBox1.Height / 4) - padding);
            this.gaugeFuel.Location = new Point((groupBox1.Width * 5 / 6 + groupBox1.Location.X), (groupBox1.Height / 4) + padding / 2 + groupBox1.Location.Y);

            this.msktxtTime.Size = new Size((groupBox1.Height / 4) - padding, (groupBox1.Height / 4) - padding);
            this.msktxtTime.Location = new Point((groupBox1.Width - msktxtTime.Width + groupBox1.Location.X - padding), (groupBox1.Location.Y + padding/2));
        }

        private void cmbxCOMPort_SelectedIndexChanged(object sender, EventArgs e)
        {
            COMPort = cmbxCOMPort.Text;
            fThread1 = false;
        }

        private void cmbxCOMPort_Click(object sender, EventArgs e)
        {
            string[] ports = SerialPort.GetPortNames();
            Array.Sort(ports);
            Array.Reverse(ports);
            cmbxCOMPort.Items.Clear();
            cmbxCOMPort.Items.AddRange(ports);
        }
    }
}